using System;
using System.Collections.Generic;

namespace Modules.DataRetrieval
{
    public static class SSEEventBus
    {
        private static readonly Dictionary<string, Action<SensorData>> SensorDataListeners = new();
        private static readonly Dictionary<string, Action<FuzzyLogicData>> FuzzyLogicDataListeners = new();

        /// <summary>
        /// Raises the appropriate event based on the type of data received.
        /// This method is generic and can handle different types of data.
        /// </summary>
        /// <typeparam name="T">The type of data being received.</typeparam>
        /// <param name="url"> The url on which the data should be sent </param>
        /// <param name="data">The data to be raised.</param>
        public static void RaiseDataReceived<T>(string url, T data)
        {
            if (typeof(T) == typeof(SensorData))
            {
                RaiseSensorData(url, data as SensorData);
            }
            else if (typeof(T) == typeof(FuzzyLogicData))
            {
                RaiseFuzzyLogicData(url, data as FuzzyLogicData);
            }
        }
    
        /// <summary>
        /// Sends the sensor data to all subscribed listeners.
        /// </summary>
        /// <param name="url">The url that is used to get the subscribers</param>
        /// <param name="data">The data to be send to the subscribers</param>
        private static void RaiseSensorData(string url, SensorData data)
        {
            if (SensorDataListeners.TryGetValue(url, out var callback))
                callback?.Invoke(data);
        }
    
        /// <summary>
        /// Sends the Fuzzy logic data to all subscribed listeners.
        /// </summary>
        /// <param name="url">The url that is used to get the subscribers</param>
        /// <param name="data">The data to be send to the subscribers</param>
        private static void RaiseFuzzyLogicData(string url, FuzzyLogicData data)
        {
            if (FuzzyLogicDataListeners.TryGetValue(url, out var callback))
                callback?.Invoke(data);
        }

        /// <summary>
        /// Subscribes to the appropriate event based on the type of data.
        /// This method allows other classes to listen for data events.
        /// </summary>
        /// <typeparam name="T">The type of data to subscribe to.</typeparam>
        /// <param name="url">The url that is subscribed to</param>
        /// <param name="callback">The callback method to be invoked when data is received.</param>
        /// <exception cref="ArgumentException">Thrown when the type of data is not supported.</exception>
        public static void Subscribe<T>(string url, Action<T> callback)
        {
            if (typeof(T) == typeof(SensorData))
                SubToSensorData(url, callback as Action<SensorData>);
            else if (typeof(T) == typeof(FuzzyLogicData))
                SubToFuzzyLogicData(url, callback as Action<FuzzyLogicData>);
        }

        /// <summary>
        /// The method subscribes to the SensorData event for the specified URL.
        /// </summary>
        /// <param name="url">The url that is subscribed to</param>
        /// <param name="callback">The callback that is assigned to the url</param>
        private static void SubToSensorData(string url, Action<SensorData> callback)
        {
            SensorDataListeners.TryAdd(url, null);
            SensorDataListeners[url] += callback;
        }
    
        /// <summary>
        /// The method subscribes to the FuzzyLogicData event for the specified URL.
        /// </summary>
        /// <param name="url">The url that is subscribed to</param>
        /// <param name="callback">The callback that is assigned to the url</param>
        private static void SubToFuzzyLogicData(string url, Action<FuzzyLogicData> callback)
        {
            FuzzyLogicDataListeners.TryAdd(url, null);
            FuzzyLogicDataListeners[url] += callback;
        }

        /// <summary>
        /// Unsubscribes from the appropriate event based on the type of data.
        /// This method allows other classes to stop listening for data events.
        /// </summary>
        /// <typeparam name="T">The type of data to unsubscribe from.</typeparam>
        /// <param name="url">The url that will be unsubscribed from</param>
        /// <param name="callback">The callback method to be removed from the event.</param>
        /// <exception cref="ArgumentException">Thrown when the type of data is not supported.</exception>
        /// <remarks>
        /// This method is used to remove a previously subscribed callback from the event.
        /// It ensures that the callback is no longer invoked when data of the specified type is received.
        /// </remarks>
        public static void Unsubscribe<T>(string url, Action<T> callback)
        {
            if (typeof(T) == typeof(SensorData))
                UnSubFromSensorData(url, callback as Action<SensorData>);
            else if (typeof(T) == typeof(FuzzyLogicData))
                UnSubFromFuzzyLogicData(url, callback as Action<FuzzyLogicData>);
        }
        
        private static void UnSubFromSensorData(string url, Action<SensorData> callback)
        {
            if (!SensorDataListeners.TryGetValue(url, out var existingCallback)) return;
        
            existingCallback -= callback;
            if (existingCallback == null)
                SensorDataListeners.Remove(url);
            else
                SensorDataListeners[url] = existingCallback;
        }
    
        private static void UnSubFromFuzzyLogicData(string url, Action<FuzzyLogicData> callback)
        {
            if (!FuzzyLogicDataListeners.TryGetValue(url, out var existingCallback)) return;
        
            existingCallback -= callback;
            if (existingCallback == null)
                FuzzyLogicDataListeners.Remove(url);
            else
                FuzzyLogicDataListeners[url] = existingCallback;
        }
    }
}